﻿using System;
using System.ComponentModel;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace PWS.API.Menu.Sample2
{
    [StructLayout(LayoutKind.Sequential)]
    public struct MenuItemInfo
    {
        public uint cbSize;
        public uint fMask;
        public uint fType;
        public uint fState;
        public uint wID;
        public IntPtr hSubMenu;
        public IntPtr hbmpChecked;
        public IntPtr hbmpUnchecked;
        public IntPtr dwItemData;
        public String dwTypeData;
        public uint cch;
        public IntPtr hbmpItem;

        // Return the size of the structure
        public static uint SizeOf
        {
            get { return (uint)Marshal.SizeOf(typeof(MenuItemInfo)); }
        }
    }

    public partial class MainForm : Form
    {
        private const int MIIM_State = 0x1;
        private const int MIIM_Id = 0x2;
        private const int MIIM_SunMenu = 0x4;
        private const int MIIM_Checkmarks = 0x8;
        private const int MIIM_Data = 0x20;
        private const int MIIM_Type = 0x100;
        private const int MIIM_String = 0x40;

        private const int MFT_String = 0x0;
        private const int MFS_Enabled = 0x0;

        private int baseCommandId = 0xf200; // Start menu id's from here
        private int currentIndex = 0;

        private const int WM_SYSCOMMAND = 0x112;

        [DllImport("user32.dll", EntryPoint = "AppendMenuA")]
        private static extern bool AppendMenu(IntPtr hMenu, MenuFlags wFlags, int wIDNewItem, string lpNewItem);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr GetSubMenu(IntPtr hMenu, int nPos);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern int GetMenuItemCount(IntPtr hMenu);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr GetMenuItemID(IntPtr hMenu, int nPosition);

        [DllImport("user32.dll", SetLastError = true)]
        public static extern IntPtr GetSystemMenu(IntPtr hwnd, bool bRevert);

        [DllImport("user32", SetLastError = true)]
        private static extern IntPtr CreatePopupMenu();

        [DllImport("user32.dll", EntryPoint = "InsertMenuItemA", SetLastError = true)]
        private static extern bool InsertMenuItem(IntPtr hMenu, int uItem, bool fByPosition, [In] ref MenuItemInfo lpmii);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool GetMenuItemInfo(IntPtr hMenu, int uItem, bool fByPosition, ref MenuItemInfo lpmii);

        [Flags]
        public enum MenuFlags : uint
        {
            MF_Checked = 0x8,
            MF_Disabled = 0x2,
            MF_Grayed = 0x1,
            MF_Separator = 0x800,
            MF_String = 0x0,
            MF_Enabled = 0x0
        }

        public MainForm()
        {
            InitializeComponent();
        }

        protected override void WndProc(ref Message m)
        {
            if (m.Msg == WM_SYSCOMMAND)
            {
                if (m.WParam.ToInt32() > baseCommandId)
                {
                    HandleSysCustomCommand(m.WParam.ToInt32());
                }
            }
            base.WndProc(ref m);
        }

        private void ShowLastError()
        {
            Win32Exception ex = new Win32Exception();
            MessageBox.Show(ex.Message, "Last Error!", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private void HandleSysCustomCommand(int cmdId)
        {
            IntPtr hMenu = GetSystemMenu(this.Handle, false);
            MenuItemInfo mnuInfo = new MenuItemInfo();
            mnuInfo.cbSize = MenuItemInfo.SizeOf;
            mnuInfo.fType = MFT_String;
            mnuInfo.fMask = MIIM_String;

            // Get size of buffer first, this call will fill out mnuInfo.cch val which is the buffer size
            if (!GetMenuItemInfo(hMenu, cmdId, false, ref mnuInfo))
            {
                ShowLastError();
            }
            else
            {
                // Pad up a bit
                mnuInfo.cch += 4;
                mnuInfo.dwTypeData = new String(' ', (Int32)mnuInfo.cch);
                if (!GetMenuItemInfo(hMenu, cmdId, false, ref mnuInfo)) // This time get menu item text
                {
                    ShowLastError();
                }
                else
                {
                    String text = String.Format("You clicked on: {0}", mnuInfo.dwTypeData);
                    MessageBox.Show(text, "Menu text", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
            }
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            IntPtr hWnd = this.Handle;
            IntPtr hMenu = GetSystemMenu(hWnd, false);

            AppendMenu(hMenu, MenuFlags.MF_Separator, 0, String.Empty);
            AppendMenu(hMenu, MenuFlags.MF_String, baseCommandId + ++currentIndex, "Item 1");
            AppendMenu(hMenu, MenuFlags.MF_Grayed | MenuFlags.MF_Disabled, baseCommandId + ++currentIndex, "Item 2");
            AppendMenu(hMenu, MenuFlags.MF_Checked, baseCommandId + ++currentIndex, "Item 3");

            IntPtr hSubMenu = CreatePopupMenu();
            AppendMenu(hSubMenu, MenuFlags.MF_String, baseCommandId + ++currentIndex, "800 * 600");
            AppendMenu(hSubMenu, MenuFlags.MF_String, baseCommandId + ++currentIndex, "1024 * 768");
            AppendMenu(hSubMenu, MenuFlags.MF_String | MenuFlags.MF_Checked, baseCommandId + ++currentIndex, "1360 * 768");

            Int32 menuItemCount = GetMenuItemCount(hMenu);

            MenuItemInfo mnuInfo = new MenuItemInfo();
            mnuInfo.cbSize = MenuItemInfo.SizeOf;
            mnuInfo.fMask = MIIM_String | MIIM_SunMenu;
            mnuInfo.fType = MFT_String;
            mnuInfo.hSubMenu = hSubMenu;
            mnuInfo.dwTypeData = "Resolution";
            InsertMenuItem(hMenu, menuItemCount, true, ref mnuInfo);

            IntPtr hResolutionItem = GetSubMenu(hMenu, menuItemCount);
            AppendMenu(hResolutionItem, MenuFlags.MF_String, baseCommandId + ++currentIndex, "1366 * 768");
        }
    }
}
